/* $Id: imaptest.c,v 1.12 1998/09/16 02:05:07 ericb Exp $ */
/* Copyright (C) 1997 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Test use of imap function */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <sys/resource.h>	/* For getrlimit */

#if 0
#include <fcntl.h>		/* For open */
#endif

#include <sicl.h>

#define	LOOP_MAX		10
#define	IOPEN_MAX		30

#define	TOTAL_PAGES		256	/* VXI A24 64 KB pages */
#define	IMAP_MAX		(TOTAL_PAGES * LOOP_MAX)

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

/* SICL-specific version of the above macro, since SICL error numbers
   can be positive.  Ick. */
#ifdef	__lint
#define	ICHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d: %s\n",\
		       progname, #func, _s, igeterrstr(igeterrno()));\
	return -1;\
    }\
} while (func)
#else
#define	ICHECK(func)	\
do {\
    int _s = (func);\
    if (_s != 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d: %s\n",\
		       progname, #func, _s, igeterrstr(igeterrno()));\
	return -1;\
    }\
} while (0)
#endif

struct map_info
{
    char   *addr;
    INST    id;
    int     map_space;
};

static const volatile char rcsid[] =
"@(#)$Id: imaptest.c,v 1.12 1998/09/16 02:05:07 ericb Exp $";
static const char *progname;

static char *
my_imap(INST id, int map_space, unsigned int pagestart,
	unsigned int pagecnt, char *suggested, int verbose)
{
    char   *rv;

    if (verbose)
    {
	(void) printf("imap(%d, %d, %u, %u, 0x%p): ",
		      id, map_space, pagestart, pagecnt, suggested);
	(void) fflush(stdout);
    }
    rv = imap(id, map_space, pagestart, pagecnt, suggested);
    if (rv == NULL)
    {
	/* Don't print for verbose == 0, because we expect to get
	   failures and don't necessarily want to clutter the output
	   with them. */
	if (verbose)
	    (void) printf("returned 0x%p: %s\n",
			  rv, igeterrstr(igeterrno()));
    }
    else if (verbose)
	(void) printf("0x%p\n", rv);

    return rv;
}

static int
my_iunmap(INST id, char *addr, int map_space, unsigned int pagestart,
	  unsigned int pagecnt, int verbose)
{
    if (verbose > 1)
    {
	(void) printf("iunmap(%d, 0x%p, %d, %u, %u): ",
		      id, addr, map_space, pagestart, pagecnt);
	(void) fflush(stdout);
    }
    ICHECK(iunmap(id, addr, map_space, pagestart, pagecnt));
    if (verbose > 1)
	(void) printf("successful\n");

    return 0;
}

static INST
my_iopen(char *addr, int verbose)
{
    INST    rv;

    if (verbose > 1)
    {
	(void) printf("iopen(\"%s\"): ", addr);
	(void) fflush(stdout);
    }
    rv = iopen(addr);
    if (rv == 0)
    {
	if (verbose > 1)
	    (void) printf("%s\n", igeterrstr(igeterrno()));
	else
	    (void) fprintf(stderr, "%s: iopen(\"%s\"): %s\n",
			   progname, addr, igeterrstr(igeterrno()));
    }
    else if (verbose > 1)
	(void) printf("%d\n", rv);

    return rv;
}

static void
print_result(char *name, int maps, unsigned int pages,
	     int window, int map_space)
{
    (void) printf("%-13s: %4d %-7s of %2u pages each in %-7s space\n",
		  name, maps, window ? "windows" : "maps",
		  pages,
		  map_space == I_MAP_A24 ? "A24" :
#ifdef	I_MAP_A24_D32
		  map_space == I_MAP_A24_D32 ? "A24_D32" :
#endif
		  "(other)");
}

#if 0
static int
eat_file_descriptors(int fildes_left)
{
    long    nfildes;
    int    *fildes;
    int     i;

    /* How many files can we open? */
    CHECK(nfildes = sysconf(_SC_OPEN_MAX));

    (void) printf("Opening %d files\n", nfildes);

    /* Allocate room for the file descriptors */
    fildes = malloc(nfildes * sizeof (*fildes));
    if (fildes == NULL)
    {
	(void) fprintf(stderr, "%s: malloc: returned NULL\n", progname);
	return -1;
    }

    /* Open them.  Some will fail because stdin, etc, have already
       been openned, so don't error if open fails. */
    for (i = 0; i < nfildes; i++)
    {
	fildes[i] = open("/dev/null", O_RDONLY);
	if (fildes[i] < 0)
	    break;
    }
    if (i != nfildes)
    {
	(void) printf("Succeeded openning %d files\n", i);
	nfildes = i;
    }
    else
    {
	/* Make sure we really can't open more */
	i = open("/dev/null", O_RDONLY);
	if (i >= 0)
	{
	    (void) fprintf(stderr, "%s: open: did not fail when expected to\n",
			   progname);
	    return -1;
	}
    }

    /* Make sure fildes_left is not too big */
    if (fildes_left > nfildes)
    {
	(void) fprintf(stderr, "%s: warning: not enough files to close\n",
		       progname);
	fildes_left = nfildes;
    }

    /* Close fildes_left files so we have that many left */
    (void) printf("Closing %d files\n", fildes_left);
    for (i = 0; i < fildes_left; i++)
	CHECK(close(fildes[i]));

    return 0;
}
#endif

/* Ask how many simultaneous A24 maps we can get on one interface */
static int
test_interface_imapinfo(int map_space, int verbose)
{
    INST    id;
    int     status, numwindows, winsize;

    id = my_iopen("vxi", verbose);
    if (id == 0)
	return -1;

    /* Initialize parameters */
    numwindows = 0;
    winsize = 0;

    /* No ICHECK for this, it can fail if map_space is unrecognised by
       the hardware */
    status = imapinfo(id, map_space, &numwindows, &winsize);
    print_result("Imapinfo", numwindows, winsize, 1, map_space);
    if (status != 0)
	(void) printf("%-13s: %s\n", "Imapinfo",
		      igeterrstr(igeterrno()));
    ICHECK(iclose(id));

    return 0;
}

static int
setup_interface(int verbose, INST *id, int iopen_count,
		struct map_info *map_info)
{
    int     i;

    /* Initialize map array */
    for (i = 0; i < IMAP_MAX; i++)
    {
	map_info[i].addr = NULL;
	map_info[i].id = 0;
	map_info[i].map_space = 0;
    }

    /* Initialize id array */
    for (i = 0; i < IOPEN_MAX; i++)
	id[i] = 0;

    /* Open VXI interfaces */
    if (iopen_count > IOPEN_MAX)
	iopen_count = IOPEN_MAX;
    for (i = 0; i < iopen_count; i++)
    {
	id[i] = my_iopen("vxi", verbose);
	if (id[i] == 0)
	    return -1;

	/* Set timeout */
	ICHECK(itimeout(id[i], 1));
    }

    return 0;
}

static int
cleanup_interface(unsigned int pagecnt, int verbose,
		  INST *id, struct map_info *map_info)
{
    int     i, loop, count;

    count = 0;
    for (loop = 0; loop < LOOP_MAX; loop++)
	for (i = 0; i < TOTAL_PAGES / pagecnt; i++)
	{
	    if (map_info[count].addr != NULL)
		CHECK(my_iunmap(map_info[count].id,
				map_info[count].addr,
				map_info[count].map_space,
				pagecnt * i, pagecnt, verbose));
	    count++;
	}

    for (i = 0; i < IOPEN_MAX; i++)
	if (id[i] != 0)
	    ICHECK(iclose(id[i]));

    return 0;
}

/* Just iclose the ids, and hope that SICL really does iunmap all the
   memory associated with these IDs. */
static int
bogus_cleanup_interface(INST *id)
{
    int     i;

    for (i = 0; i < IOPEN_MAX; i++)
	if (id[i] != 0)
	    ICHECK(iclose(id[i]));

    return 0;
}

/* Open simultaneous A24 maps on one interface */
static int
run_interface_max(int map_space, unsigned int pagecnt, int verbose,
		  INST *id, struct map_info *map_info, int offset)
{
    int     i, loop, count, max_map;

    /* Initialize variables */
    max_map = -1;

    /* See how many imaps we can do */
    count = 0;
    for (loop = 0; loop < LOOP_MAX; loop++)
    {
	for (i = 0; i < TOTAL_PAGES / pagecnt; i++)
	{
	    if (count < offset)
	    {
		count++;
		continue;
	    }

	    map_info[count].id = id[0];
	    map_info[count].map_space = map_space;
	    map_info[count].addr = my_imap(id[0], map_space,
					   pagecnt * i, pagecnt,
					   NULL, verbose);

	    /* If two failures, quit now */
	    if (map_info[count].addr == NULL && max_map >= 0)
		break;

	    /* Remember the first failure */
	    if (map_info[count].addr == NULL && max_map == -1)
		max_map = count;

	    count++;
	}
	if (max_map != -1)
	    break;
    }
    if (max_map == -1)
	max_map = IMAP_MAX / pagecnt;
    max_map -= offset;

    print_result("One ID", max_map, pagecnt, 0, map_space);

    return 0;
}

/* Open simultaneous A24 maps on multiple opens of one interface */
static int
run_interface_reopen(int map_space, unsigned int pagecnt, int verbose,
		     INST *id, struct map_info *map_info, int offset)
{
    int     i, loop, count, max_map, id_count;

    /* Initialize variables */
    max_map = -1;
    id_count = 0;

    if (offset > 0)
	id_count += 5;

    /* See how many imaps we can do */
    count = 0;
    for (loop = 0; loop < LOOP_MAX; loop++)
    {
	for (i = 0; i < TOTAL_PAGES / pagecnt; i++)
	{
	    if (count < offset)
	    {
		count++;
		continue;
	    }

	    map_info[count].id = id[id_count];
	    map_info[count].map_space = map_space;
	    map_info[count].addr = my_imap(id[id_count], map_space,
					   pagecnt * i, pagecnt,
					   NULL, verbose);

	    if (map_info[count].addr == NULL)
	    {
		/* The imap failed, so try again with next id */
		id_count++;
		if (id_count >= IOPEN_MAX)
		{
		    /* No more ids, stop now */
		    if (max_map == -1)
			max_map = count;
		    break;
		}

		/* Retry imap */
		map_info[count].id = id[id_count];
		map_info[count].addr = my_imap(id[id_count],
					       map_space, pagecnt * i,
					       pagecnt, NULL,
					       verbose);
	    }

	    /* If two failures, quit now */
	    if (map_info[count].addr == NULL && max_map >= 0)
		break;

	    /* Remember the first failure */
	    if (map_info[count].addr == NULL && max_map == -1)
		max_map = count;

	    count++;
	}
	if (max_map != -1)
	    break;
    }
    if (max_map == -1)
	max_map = IMAP_MAX / pagecnt;
    max_map -= offset;

    print_result("Multiple ID", max_map, pagecnt, 0, map_space);

    return 0;
}

/* Open A24 maps on one interface, closing old ones when a failure is
   found. */
static int
run_interface_reuse(int map_space, unsigned int pagecnt, int verbose,
		    INST *id, struct map_info *map_info)
{
    int     i, loop, count, subi, subloop, subcount, max_map;

    /* Initialize variables */
    max_map = -1;

    /* See how many imaps we can do */
    count = 0;
    for (loop = 0; loop < LOOP_MAX; loop++)
    {
	for (i = 0; i < TOTAL_PAGES / pagecnt; i++)
	{
	    map_info[count].id = id[0];
	    map_info[count].map_space = map_space;
	    map_info[count].addr = my_imap(id[0], map_space,
					   pagecnt * i, pagecnt,
					   NULL, verbose);

	    if (map_info[count].addr == NULL)
	    {
		/* The imap failed, so unmap a previous imap and retry */

		/* Find previous successful imap */ 
		subcount = 0;
		for (subloop = 0; subloop < LOOP_MAX; subloop++)
		{
		    for (subi = 0; subi < TOTAL_PAGES / pagecnt; subi++)
		    {
			if (map_info[subcount].addr != NULL)
			    break;
			subcount++;
		    }
		    if (subi != TOTAL_PAGES / pagecnt)
			break;
		}
		if (subcount == IMAP_MAX / pagecnt)
		{
		    /* No more previous maps left, stop now */
		    if (max_map == -1)
			max_map = count;
		    break;
		}

		/* Unmap it */
		CHECK(my_iunmap(map_info[subcount].id,
				map_info[subcount].addr,
				map_info[subcount].map_space,
				pagecnt * subi, pagecnt, verbose));
		map_info[subcount].addr = NULL;

		/* Retry imap */
		map_info[count].addr = my_imap(id[0], map_space,
					       pagecnt * i,
					       pagecnt, NULL,
					       verbose);
	    }

	    /* If two failures, quit now */
	    if (map_info[count].addr == NULL && max_map >= 0)
		break;

	    /* Remember the first failure */
	    if (map_info[count].addr == NULL && max_map == -1)
		max_map = count;

	    count++;
	}
	if (max_map != -1)
	    break;
    }
    if (max_map == -1)
	max_map = IMAP_MAX / pagecnt;

    print_result("Reusing Maps", max_map, pagecnt, 0, map_space);

    return 0;
}

/* Test how many simultaneous A24 maps we can get on one interface */
static int
test_interface_max(int map_space, unsigned int pagecnt, int verbose)
{
    INST    id[IOPEN_MAX];
    struct map_info map_info[IMAP_MAX];

    CHECK(setup_interface(verbose, id, 1, map_info));
    CHECK(run_interface_max(map_space, pagecnt, verbose,
			    id, map_info, 0));
    CHECK(cleanup_interface(pagecnt, verbose, id, map_info));

    return 0;
}

/* Test how many simultaneous A24 maps we can get on one interface,
   trying two different map spaces at the same time. */
static int
test_interface_max2(int map_space1, int map_space2,
		    unsigned int pagecnt, int verbose)
{
    INST    id[IOPEN_MAX];
    struct map_info map_info[IMAP_MAX];

    CHECK(setup_interface(verbose, id, 1, map_info));
    CHECK(run_interface_max(map_space1, pagecnt, verbose,
			    id, map_info, 0));
    CHECK(run_interface_max(map_space2, pagecnt, verbose,
			    id, map_info, 40));
    CHECK(cleanup_interface(pagecnt, verbose, id, map_info));

    return 0;
}

/* Test how many simultaneous A24 maps on multiple opens of one interface */
static int
test_interface_reopen(int map_space, unsigned int pagecnt,
		      int verbose)
{
    INST    id[IOPEN_MAX];
    struct map_info map_info[IMAP_MAX];

    CHECK(setup_interface(verbose, id, IOPEN_MAX, map_info));
    CHECK(run_interface_reopen(map_space, pagecnt, verbose,
			       id, map_info, 0));
    CHECK(cleanup_interface(pagecnt, verbose, id, map_info));

    return 0;
}

/* Test how many simultaneous A24 maps on multiple opens of one interface,
   trying two different map spaces at the same time. */
static int
test_interface_reopen2(int map_space1, int map_space2,
		       unsigned int pagecnt, int verbose)
{
    INST    id[IOPEN_MAX];
    struct map_info map_info[IMAP_MAX];

    CHECK(setup_interface(verbose, id, IOPEN_MAX, map_info));
    CHECK(run_interface_reopen(map_space1, pagecnt, verbose,
			       id, map_info, 0));
    CHECK(run_interface_reopen(map_space2, pagecnt, verbose,
			       id, map_info, 40));
    CHECK(cleanup_interface(pagecnt, verbose, id, map_info));

    return 0;
}

/* Test how many simultaneous A24 maps on multiple opens of one interface,
   trying two different map spaces at the same time, and falling back
   to smaller pagecount for the second try. */
static int
test_interface_reopen3(int map_space1, int map_space2, int verbose)
{
    INST    id[IOPEN_MAX];
    struct map_info map_info[IMAP_MAX];

    CHECK(setup_interface(verbose, id, IOPEN_MAX, map_info));
    CHECK(run_interface_reopen(map_space1, 2, verbose,
			       id, map_info, 0));
    CHECK(run_interface_reopen(map_space2, 1, verbose,
			       id, map_info, 40));
    CHECK(bogus_cleanup_interface(id));

    return 0;
}

/* Test how many A24 maps we can get if we unmap an old one when a
   failure is found. */
static int
test_interface_reuse(int map_space, unsigned int pagecnt, int verbose)
{
    INST    id[IOPEN_MAX];
    struct map_info map_info[IMAP_MAX];

    CHECK(setup_interface(verbose, id, 1, map_info));
    CHECK(run_interface_reuse(map_space, pagecnt, verbose,
			      id, map_info));
    CHECK(cleanup_interface(pagecnt, verbose, id, map_info));

    return 0;
}

static int
run(int verbose)
{
    (void) printf("Starting IMAP Test\n");

    CHECK(test_interface_imapinfo(I_MAP_A24, verbose));
#ifdef	I_MAP_A24_D32
    CHECK(test_interface_imapinfo(I_MAP_A24_D32, verbose));
#endif
    CHECK(test_interface_max(I_MAP_A24, 1, verbose));
    CHECK(test_interface_max(I_MAP_A24, 2, verbose));
#ifdef	I_MAP_A24_D32
    CHECK(test_interface_max(I_MAP_A24_D32, 1, verbose));
    CHECK(test_interface_max(I_MAP_A24_D32, 2, verbose));
    CHECK(test_interface_max2(I_MAP_A24_D32, I_MAP_A24, 1, verbose));
    CHECK(test_interface_max2(I_MAP_A24_D32, I_MAP_A24, 2, verbose));
#endif
    CHECK(test_interface_reopen(I_MAP_A24, 1, verbose));
    CHECK(test_interface_reopen(I_MAP_A24, 2, verbose));
#ifdef	I_MAP_A24_D32
    CHECK(test_interface_reopen(I_MAP_A24_D32, 1, verbose));
    CHECK(test_interface_reopen(I_MAP_A24_D32, 2, verbose));
    CHECK(test_interface_reopen2(I_MAP_A24_D32, I_MAP_A24, 1, verbose));
    CHECK(test_interface_reopen2(I_MAP_A24_D32, I_MAP_A24, 2, verbose));
    CHECK(test_interface_reopen3(I_MAP_A24, I_MAP_A24_D32, verbose));
    CHECK(test_interface_reopen3(I_MAP_A24_D32, I_MAP_A24, verbose));
#endif
    CHECK(test_interface_reuse(I_MAP_A24, 1, verbose));
    CHECK(test_interface_reuse(I_MAP_A24, 2, verbose));
#ifdef	I_MAP_A24_D32
    CHECK(test_interface_reuse(I_MAP_A24_D32, 1, verbose));
    CHECK(test_interface_reuse(I_MAP_A24_D32, 2, verbose));
#endif

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
	"Usage: %s [-uvV]\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    int     c, verbose;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "uvV")) != -1)
	switch (c)
	{
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    (void) system("date");
    (void) system("uname -a");
    (void) system("what /usr/lib/libsicl.sl");

    /* Run the tests, twice to be sure */
    if (run(verbose) < 0)
	return EXIT_FAILURE;

#ifdef	RLIMIT_NOFILE
    {
	/* Attempt to increase the number of file descriptors
	   available, in case that helps. */
	struct rlimit rl;
	if (getrlimit(RLIMIT_NOFILE, &rl) == 0)
	{
	    rl.rlim_cur = rl.rlim_max;
	    (void) setrlimit(RLIMIT_NOFILE, &rl);
	}
    }
#endif

    if (run(verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
